#Requires -Version 5.1

Set-StrictMode -Version Latest
$ErrorActionPreference = "Stop"

# =========================
# Config (PS5 compatible)
# =========================
$BASE_URL       = if ($env:BASE_URL)       { $env:BASE_URL }       else { "http://<BASE_URL>" }
$API_KEY        = if ($env:API_KEY)        { $env:API_KEY }        else { "<API_KEY>" }
$MODEL_NAME     = if ($env:MODEL_NAME)     { $env:MODEL_NAME }     else { "<MODEL_NAME>" }
$EMB_MODEL_NAME = if ($env:EMB_MODEL_NAME) { $env:EMB_MODEL_NAME } else { "<EMB_MODEL_NAME>" }
$IMAGE_URL      = if ($env:IMAGE_URL)      { $env:IMAGE_URL }      else { "" }

$TIMEOUT = if ($env:CURL_MAX_TIME) { [int]$env:CURL_MAX_TIME } else { 60 }

# =========================
# Global stats
# =========================
$FAILED_COUNT  = 0
$FAILED_TITLES = @()
$FAILED_DETAILS = @()

# =========================
# Helpers
# =========================

function Has-Property($obj, $name) {
    return $obj.PSObject.Properties.Name -contains $name
}

function Fail($title, $detail) {
    $script:FAILED_COUNT++
    $script:FAILED_TITLES  += $title
    $script:FAILED_DETAILS += $detail

    Write-Host ""
    Write-Host "===================="
    Write-Host "[ERROR] $title"
    Write-Host "--------------------"
    Write-Host $detail
    Write-Host "===================="
    Write-Host ""
}

function Pass($msg) {
    Write-Host "[OK] $msg"
}

function Http-PostJson($url, $bodyObj) {

    $json = $bodyObj | ConvertTo-Json -Depth 10

    try {
        $resp = Invoke-WebRequest `
            -Uri $url `
            -Method Post `
            -Headers @{
                "Content-Type" = "application/json"
                "Authorization" = "Bearer $API_KEY"
            } `
            -Body $json `
            -TimeoutSec $TIMEOUT

        return @{
            body = $resp.Content
            code = $resp.StatusCode
        }
    }
    catch {
        Fail "Network error" "$($_.Exception.Message)`nURL=$url"
        return $null
    }
}

# =========================
# 1) Chat Completions check
# =========================
function Check-Chat {

    $url = "$($BASE_URL.TrimEnd('/'))/chat/completions"

    $payload = @{
        model = $MODEL_NAME
        messages = @(
            @{ role="user"; content="Hello" }
        )
        stream = $false
    }

    $resp = Http-PostJson $url $payload
    if (-not $resp) { return }

    if ($resp.code -ne 200) {
        Fail "Chat endpoint HTTP != 200" "HTTP=$($resp.code)`n$($resp.body)"
        return
    }

    $obj = $resp.body | ConvertFrom-Json

    if (Has-Property $obj "error") {
        Fail "Chat returned error" ($obj.error | ConvertTo-Json -Depth 5)
        return
    }

    if (-not (Has-Property $obj "choices")) {
        Fail "Chat response no choices field" $resp.body
        return
    }

    $content = $obj.choices[0].message.content

    if (-not $content) {
        Fail "Chat content is empty or response structure is unexpected" $resp.body
        return
    }

    Pass "Chat Completions OK (content is not empty)"
}

# =========================
# 2) Embeddings check
# =========================
function Check-Embeddings {

    $url = "$($BASE_URL.TrimEnd('/'))/embeddings"

    $payload = @{
        model = $EMB_MODEL_NAME
        input = "test"
    }

    $resp = Http-PostJson $url $payload
    if (-not $resp) { return }

    if ($resp.code -ne 200) {
        Fail "Embeddings endpoint HTTP != 200" "HTTP=$($resp.code)`n$($resp.body)"
        return
    }

    $obj = $resp.body | ConvertFrom-Json

    if (Has-Property $obj "error") {
        Fail "Embeddings returned error" ($obj.error | ConvertTo-Json -Depth 5)
        return
    }

    if (-not (Has-Property $obj "data")) {
        Fail "Embeddings no data field" $resp.body
        return
    }

    $len = 0
    if ($obj.data[0] -and (Has-Property $obj.data[0] "embedding")) {
        $len = $obj.data[0].embedding.Length
    }

    if ($len -le 0) {
        Fail "Embeddings vector is empty or structure is unexpected" $resp.body
        return
    }

    Pass "Embeddings OK (embedding_length=$len)"
}

# =========================
# 3) Tools check
# =========================
function Check-Tools {

    $url = "$($BASE_URL.TrimEnd('/'))/chat/completions"

    $payload = @{
        model = $MODEL_NAME
        stream = $false
        messages = @(
            @{ role="system"; content="You can call tools when needed." },
            @{ role="user"; content="Call get_time tool." }
        )
        tools = @(
            @{
                type = "function"
                function = @{
                    name = "get_time"
                    description = "Get current time"
                    parameters = @{
                        type="object"
                        properties=@{}
                    }
                }
            }
        )
    }

    $resp = Http-PostJson $url $payload
    if (-not $resp) { return }

    $obj = $resp.body | ConvertFrom-Json

    if (Has-Property $obj "error") {
        Fail "Tools returned error" ($obj.error | ConvertTo-Json -Depth 5)
        return
    }

    $tcLen = 0
    if (Has-Property $obj.choices[0].message "tool_calls") {
        $tcLen = ($obj.choices[0].message.tool_calls | Measure-Object).Count
    }

    $finish = $obj.choices[0].finish_reason

    if ($tcLen -le 0 -and $finish -ne "tool_calls") {
        Fail "Tools check failed" $resp.body
        return
    }

    Pass "Tools OK (tool_calls_length=$tcLen finish_reason=$finish)"
}

# =========================
# 4) Image check
# =========================
function Check-Image {

    if (-not $IMAGE_URL) {
        Write-Host "[INFO] Skip image check (IMAGE_URL not set)"
        return
    }

    $url = "$($BASE_URL.TrimEnd('/'))/chat/completions"

    $payload = @{
        model = $MODEL_NAME
        stream = $false
        messages = @(
            @{
                role="user"
                content=@(
                    @{ type="text"; text="Describe the image in 20 words." },
                    @{ type="image_url"; image_url=@{ url=$IMAGE_URL } }
                )
            }
        )
    }

    $resp = Http-PostJson $url $payload
    if (-not $resp) { return }

    $obj = $resp.body | ConvertFrom-Json

    if (Has-Property $obj "error") {
        Fail "Image request returned error (image may be inaccessible or model lacks multimodal)" ($obj.error | ConvertTo-Json -Depth 5)
        return
    }

    $content = $obj.choices[0].message.content

    if (-not $content) {
        Fail "Image response content is empty or structure is unexpected" $resp.body
        return
    }

    Pass "Image request OK (model can access image)"
}

# =========================
# Main
# =========================

Write-Host "BASE_URL=$BASE_URL"
Write-Host "MODEL_NAME=$MODEL_NAME"
Write-Host "EMB_MODEL_NAME=$EMB_MODEL_NAME"
if ($IMAGE_URL) {
    Write-Host "IMAGE_URL=$IMAGE_URL"
}
Write-Host ""

Check-Chat
Check-Embeddings
Check-Tools
Check-Image

Write-Host ""

if ($FAILED_COUNT -gt 0) {

    Write-Host "[WARNING] Completed with $FAILED_COUNT failure(s)"

    for ($i=0; $i -lt $FAILED_TITLES.Count; $i++) {
        Write-Host "$($i+1)) $($FAILED_TITLES[$i])"
        Write-Host $FAILED_DETAILS[$i]
        Write-Host ""
    }

    exit 1
}

Write-Host "All checks passed"
exit 0
